import mysql.connector
import spacy
import re
import sys
import unicodedata

# Cargar el modelo de lenguaje en español de spaCy (modelo grande)
nlp = spacy.load("es_core_news_lg")

# Configuración de la conexión a la base de datos
db_config = {
    'user': 'advs_14',
    'password': 'advperro2',
    'host': 'localhost',
    'database': 'advs_14'
}

def extraer_nombres(texto):
    """Extrae nombres de personas de un texto usando spaCy, palabras en mayúsculas completas,
    y palabras que comiencen con mayúscula."""
    doc = nlp(texto)
    
    # Extraer entidades nombradas de tipo 'PER' (persona)
    nombres_personas = [ent.text for ent in doc.ents if ent.label_ == "PER"]
    
    # Filtrar solo las palabras en mayúsculas completas
    palabras_mayusculas = re.findall(r'\b[A-ZÁÉÍÓÚÑ][A-ZÁÉÍÓÚÑ]*(?: [A-ZÁÉÍÓÚÑ][A-ZÁÉÍÓÚÑ]*)*\b', texto)
    
    # Filtrar palabras que inicien con una letra mayúscula
    palabras_con_mayuscula_inicial = re.findall(r'\b[A-ZÁÉÍÓÚÑ][a-záéíóúñ]*\b', texto)
    
    # Combinar y deduplicar todos los nombres y palabras
    nombres_combined = list(set(nombres_personas + palabras_mayusculas + palabras_con_mayuscula_inicial))

    #imprimir nombres 
    print(nombres_combined)

    
    return nombres_combined

# Ejemplo de uso


def obtener_apellidos(cursor):
    """Obtiene la lista de apellidos de la tabla 'nombres'."""
    cursor.execute("SELECT name FROM advs_14.nombres WHERE LENGTH(name) > 4;")
    rows = cursor.fetchall()
    apellidos = [row[0] for row in rows]
    return apellidos

def normalizar_texto(texto):
    """Elimina acentos y transforma el texto a minúsculas."""
    texto = unicodedata.normalize('NFKD', texto)
    texto = ''.join(c for c in texto if not unicodedata.combining(c))
    return texto.lower()

def nombre_contiene_apellido(nombre, apellidos):
    """Verifica si el nombre contiene al menos dos apellidos diferentes de la lista."""
    # Normalizar el nombre
    nombre_normalizado = normalizar_texto(nombre)
    nombre_parts = set(nombre_normalizado.split())

    apellidos_encontrados = set()  # Utilizamos un conjunto para almacenar apellidos encontrados

    for apellido in apellidos:
        apellido_normalizado = normalizar_texto(apellido)
        if apellido_normalizado in nombre_parts:
            apellidos_encontrados.add(apellido_normalizado)
        if len(apellidos_encontrados) >= 2:
            return True
            
    return False

def nombre_similar_existe(nombre, nombres_existentes):
    """Verifica si un nombre similar ya existe en la lista de nombres existentes y reemplaza el más corto por el más largo.
    También devuelve True si la primera palabra encontrada en ambas comparaciones es distinta.
    """
    nombre_lower = nombre.lower()
    primera_palabra_nombre = nombre_lower.split()[0]  # Obtiene la primera palabra del nombre

    for i, nombre_existente in enumerate(nombres_existentes):
        nombre_existente_lower = nombre_existente.lower()
        primera_palabra_nombre_existente = nombre_existente_lower.split()[0]  # Obtiene la primera palabra del nombre existente
        
        if primera_palabra_nombre != primera_palabra_nombre_existente:
            return True

        if nombre_lower == nombre_existente_lower:
            # Si el nombre nuevo es más largo, reemplaza el existente
            if len(nombre) > len(nombre_existente):
                nombres_existentes[i] = nombre
            return True

    return False

def reemplazar_y_espacios(texto):
    """Reemplaza ' Y ' por ',' y dos espacios por ',' en el texto dado."""
    texto = texto.replace(' Y ', ',')
    texto = re.sub(r'\s{2,}', ' , ', texto)  # Reemplaza dos o más espacios por una coma
    return texto
    
def limpiar_nombres_largos(nombres, apellidos):
    """
    Recorta nombres que contienen apellidos de una lista específica, limitando a 2 palabras antes y 1 después del apellido.

    Args:
    - nombres: Lista de nombres identificados.
    - apellidos: Lista de apellidos para buscar.

    Returns:
    - Lista de nombres limpios.
    """
    nombres_limpios = []

    for nombre in nombres:
        palabras = nombre.split()

        # Comprobar si el nombre tiene más de 5 palabras
        if len(palabras) > 4:
            for i, palabra in enumerate(palabras):
                if palabra in apellidos:
                    # Recortar el nombre a 2 palabras antes y 1 después del apellido
                    inicio = max(0, i - 2)  # No ir antes del inicio de la lista
                    fin = min(len(palabras), i + 2)  # Incluir el apellido y una palabra después
                    nombre_recortado = ' '.join(palabras[inicio:fin])
                    nombres_limpios.append(nombre_recortado)
                    break
            else:
                # Si no se encuentra ningún apellido, añadir el nombre original
                nombres_limpios.append(nombre)
        else:
            # Añadir el nombre original si tiene 5 o menos palabras
            nombres_limpios.append(nombre)

    return nombres_limpios


def limpiar_nombres(nombres):
    """Limpia la lista de nombres eliminando duplicados y nombres similares,
    excepto cuando los nombres marcados como iguales comienzan con una palabra distinta.
    """
    # Descomponer nombres en listas de palabras
    nombres_descompuestos = [(nombre, nombre.split()) for nombre in nombres]

    # Listas para almacenar nombres que deben ser eliminados y los pares considerados como iguales
    nombres_a_eliminar = set()
    pares_iguales = set()

    # Comparar cada nombre con los demás
    for i, (nombre1, partes1) in enumerate(nombres_descompuestos):
        for j, (nombre2, partes2) in enumerate(nombres_descompuestos):
            if i >= j:  # Evitar comparar el mismo nombre o repetir comparaciones
                continue

            # Contar coincidencias entre partes de los nombres
            coincidencias = len(set(partes1) & set(partes2))

            if coincidencias >= 2:
                # Verificar si ya se han marcado como iguales previamente
                if (nombre1, nombre2) in pares_iguales or (nombre2, nombre1) in pares_iguales:
                    # Si ya se marcaron como iguales, no hacer nada
                    continue

                # Marcar los nombres como iguales
                pares_iguales.add((nombre1, nombre2))
                
                # Si hay coincidencias suficientes y comienzan con la misma palabra
                if partes1[0] == partes2[0]:
                    # Marcar el nombre más corto para eliminar
                    if len(nombre1) <= len(nombre2):
                        nombres_a_eliminar.add(nombre1)
                    else:
                        nombres_a_eliminar.add(nombre2)

    # Retornar la lista de nombres limpios
    nombres_limpios = [nombre for nombre in nombres if nombre not in nombres_a_eliminar]
    return nombres_limpios

def main(id_doc):
    # Conectar a la base de datos
    connection = mysql.connector.connect(**db_config)
    cursor = connection.cursor()

    # Obtener apellidos de la tabla 'nombres'
    apellidos = obtener_apellidos(cursor)

    # Seleccionar datos de la tabla 'data' usando el parámetro id_doc
    cursor.execute("""
        SELECT id, raw_data, ruta_doc, id_documento, procesado, proc_data
        FROM advs_14.data
        WHERE procesado = 0 AND id_documento = %s;
    """, (id_doc,))
    rows = cursor.fetchall()

    # Lista para almacenar nombres ya identificados
    nombres_existentes = []

    for row in rows:
        id_data = row[0]
        raw_data = row[1]
        id_documento = row[3]

        # Reemplazar ' Y ' y dos espacios por comas en el raw_data
        raw_data_reemplazado = reemplazar_y_espacios(raw_data)

        # Extraer nombres de personas del campo 'raw_data' modificado
        nombres_identificados = extraer_nombres(raw_data_reemplazado)

        # Limpiar nombres identificados
        nombres_identificados = limpiar_nombres(nombres_identificados)

        # Limpiar nombres largos
        nombres_identificados = limpiar_nombres_largos(nombres_identificados, apellidos)

        for nombre in nombres_identificados:
            if nombre_contiene_apellido(nombre, apellidos):
                    try:
                        # Insertar nombres en la tabla 'interesados'
                        cursor.execute("""
                            INSERT INTO advs_14.interesados (nombre, id_documento, id_pagina)
                            VALUES (%s, %s, %s)
                        """, (nombre, id_documento, id_data))
                        connection.commit()

                        # Añadir el nombre a la lista de nombres ya insertados
                        nombres_existentes.append(nombre)
                    except mysql.connector.Error as err:
                        print(f"Error al insertar {nombre}: {err}")

    # Cerrar la conexión a la base de datos
    cursor.close()
    connection.close()

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Uso: python3 /var/www/html/scaner/tensor3.py <id_doc>")
        sys.exit(1)

    id_doc = sys.argv[1]
    main(id_doc)
