import os
import re
import hashlib
import mysql.connector
import pytesseract
from PIL import Image

# Configuración de la base de datos
db_config = {
    'user': 'advs',
    'password': 'advperro2',
    'host': 'localhost',
    'database': 'advs'
}

# Archivo de bloqueo
lock_file = "/tmp/process_images.lock"

# Función para bloquear el script
def check_lock():
    if os.path.exists(lock_file):
        print("El script ya está en ejecución.")
        exit(1)
    else:
        open(lock_file, 'w').close()

# Función para liberar el bloqueo
def release_lock():
    if os.path.exists(lock_file):
        os.remove(lock_file)

# Conectar a la base de datos
def connect_db():
    return mysql.connector.connect(**db_config)

# Extraer ruta del documento a procesar
def get_document_to_process(connection):
    cursor = connection.cursor()
    query = """
    SELECT id, ruta, fecha_subida, procesado, nombre_original, ip_proc, aprovado, procesado_meta
    FROM advs.documentos
    WHERE procesado = 1 AND procesado_meta = 0
    LIMIT 1;
    """
    cursor.execute(query)
    result = cursor.fetchone()
    cursor.close()
    return result

# Leer archivo imgs.txt y procesar imágenes
def process_images(imgs_file_path):
    if not os.path.exists(imgs_file_path):
        print(f"El archivo {imgs_file_path} no existe.")
        return []
    
    with open(imgs_file_path, 'r') as f:
        image_paths = [line.strip() for line in f.readlines()]
    
    ocr_results = []
    for img_path in image_paths:
        print(f"Procesando imagen: {img_path}")
        if os.path.exists(img_path):
            text = pytesseract.image_to_string(Image.open(img_path))
            ocr_results.append((img_path, text))
        else:
            print(f"La imagen {img_path} no existe.")
    
    return ocr_results

# Limpiar el texto para solo tener letras, números y un solo espacio entre palabras
def clean_text(text):
    # Eliminar cualquier carácter que no sea letra, número o espacio
    cleaned_text = re.sub(r'[^A-Za-z0-9\s]', '', text)
    # Reemplazar múltiples espacios consecutivos por un solo espacio
    cleaned_text = re.sub(r'\s+', ' ', cleaned_text)
    return cleaned_text.strip().lower()  # Convertir a minúsculas

# Encriptar las palabras en SHA1 y reemplazar los espacios con saltos de línea
def encrypt_words(text):
    words = text.split()  # Separar por espacios
    encrypted_words = [hashlib.sha1(word.encode('utf-8')).hexdigest() for word in words]
    return '\n'.join(encrypted_words)  # Reemplazar espacios por saltos de línea

# Insertar resultados en la tabla `data`
def insert_data(connection, id_documento, ocr_results):
    cursor = connection.cursor()
    for img_path, text in ocr_results:
        cleaned_text = clean_text(text)  # Limpiar y convertir el texto a minúsculas
        encrypted_data = encrypt_words(cleaned_text)  # Encriptar las palabras y formatearlas
        
        query = """
        INSERT INTO advs.data (raw_data, ruta_doc, id_documento, procesado, proc_data)
        VALUES (%s, %s, %s, 1, %s);
        """
        cursor.execute(query, (text, img_path, id_documento, encrypted_data))
    connection.commit()
    cursor.close()

# Actualizar el campo procesado_meta del documento
def update_document_meta(connection, id_documento):
    cursor = connection.cursor()
    query = """
    UPDATE advs.documentos
    SET procesado_meta = 1
    WHERE id = %s;
    """
    cursor.execute(query, (id_documento,))
    connection.commit()
    cursor.close()

# Ejecutar el script principal
def main():
    check_lock()
    
    connection = connect_db()
    
    try:
        # Obtener el documento a procesar
        document = get_document_to_process(connection)
        if document:
            id_documento, ruta, *_ = document
            
            # Cambiar la ruta al archivo imgs.txt
            base_path = os.path.dirname(ruta)
            imgs_file_path = os.path.join(base_path, "imgs.txt")
            
            # Procesar las imágenes listadas en imgs.txt
            ocr_results = process_images(imgs_file_path)
            
            # Insertar los resultados en la base de datos
            if ocr_results:
                insert_data(connection, id_documento, ocr_results)
                # Actualizar el campo procesado_meta
                update_document_meta(connection, id_documento)
                print(f"Documento {id_documento} procesado y meta actualizada.")
            else:
                print("No se encontraron resultados OCR.")
        else:
            print("No hay documentos para procesar.")
    
    finally:
        connection.close()
        release_lock()

if __name__ == "__main__":
    main()
